********************************************************************************
********************************************************************************
* Date: November 2025
** Do: Code to reproduce Chapter 1 of World Inequality Report 2026
*** Author: Ricardo Gómez-Carrera
	* 1. Figure 1.1. The world is becoming richer
	* 2. Figure 1.2. Poorest half of the world population: 2.8 billion adults
	* 3. Figure 1.3. Income and wealth shares are distributed very unequally
	* 4. Figure 1.4. Income grows faster at the top
	* 5. Figure 1.5. Income is growing the least for the global middle class
	* 6. Figure 1.6. Wealth is increasing much more at the very top
	* 7. Figure 1.7. Wealth grows faster among the very wealthy
	* 8. Figure 1.8. Extreme wealth inequality is persistent and increasing
	* 9. Figure 1.9. Income inequality has persisted for centuries
	*10. Figure 1.10. Extreme income inequality has been persistent during the last two centuries
	*11. Figure 1.11. Uneven repartition of income
	*12. Figure 1.12. There is very large inequality across regions
	*13. Figure 1.13. There is also very large inequality within regions
	*14. Figure Box 1.2: Inequality Transparency Index
********************************************************************************
********************************************************************************

/*
For information on how to use the widcode and the World Inequality Database, visit:
	*1. https://wid.world/codes-dictionary/
	*2. https://github.com/world-inequality-database/wid-stata-tool
	*3. https://github.com/world-inequality-database/wid-r-tool
	*4. https://wid.world/document/distributional-national-accounts-dina-guidelines-2025-methods-and-concepts-used-in-the-world-inequality-database/
	
	For other data inputs visit:
	*1. https://github.com/world-inequality-database/wid-world/tree/master/data-input
	
*/

**# Figure 1.1. The world is becoming richer
* ---------------------------------------------------------------------------- 
*See WIR2026_Chapter2.do, Figure 2.1 
* ---------------------------------------------------------------------------- * 

**# Figure 1.2. Poorest half of the world population: 2.8 billion adults
* ---------------------------------------------------------------------------- 
* ---------------------------------------------------------------------------- * 
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(npopul) age(999 992) area(WO) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "npopul999i", "npopul992i") 
}
else{
use if 	inlist(iso, "WO") & year == $year_output & ///
		inlist(widcode, "npopul999i", "npopul992i") ///
		using $wid2024, clear
}	
drop currency


keep if year==$year_output

gen bot50 = value*.5/1e9
gen mid40 = value*.4/1e9  
gen top10 = value*.1/1e6  
gen top1  = value*.01/1e6
gen top_1  = value*.001/1e6 
gen top_01  = value*.0001
gen top_001  = value*.00001
gen top_0001  = value*.000001
gen top_00001  = value*.0000001
gen top_000001  = value*.00000001

rename value total
label var total "Total"
label var bot50 "Bottom 50%"
label var mid40 "Middle 40%"
label var top10 "Top 10%"
label var top1  "Top 1%"
label var top_1  "Top 0.1%"
label var top_01  "Top 0.01%"
label var top_001  "Top 0.001%"
label var top_0001  "Top 0.0001%"
label var top_00001  "Top 0.00001%"
label var top_000001  "Top 0.000001%"
drop total
replace widcode = "Total Population" if widcode == "npopul999i" 
replace widcode = "Adult Population" if widcode == "npopul992i" 
drop iso p year
g year=$year_output
export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.2") sheetmod cell(A2) first(varl) 





**# Figure 1.3. Income and wealth shares are distributed very unequally 
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(sptinc shweal) age(992) area(WO) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
}
else{
use if inlist(iso, "WO") & year == $year_output ///
	using $wid2024, clear
}	




drop currency iso year 
keep if inlist(widcode, "sptinc992j", "shweal992j") ///
	  & inlist(p, "p0p50", "p50p90", "p90p100", "p99p100")

reshape wide value, i(widcode) j(p) string

renvars value*, pred(5)
label var p0p50   "Bottom 50%"
label var p50p90  "Middle 40%"
label var p90p100 "Top 10%"
label var p99p100 "Top 1%"
label var widcode " "

replace widcode = "Income" if widcode == "sptinc992j" 
replace widcode = "Wealth" if widcode == "shweal992j" 
gsort widcode

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.3.a") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.3.a")
putexcel (B3:E4), nformat(percent)




**# Figure 1.4. Income grows faster at the top   
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(aptinc) age(992) pop(j) area( WO WO-MER ) year( 1980 $year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"aptinc992j")
}
else{
use if widcode == "aptinc992j" & (year == $year_output | year == 1980) & (iso == "WO" | iso == "WO-MER") using $wid2024, clear
}	

generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
generate long p_max = round(1000*real(regexs(2))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")

generate n = round(p_max - p_min, 1)
keep if inlist(n, 1, 10, 100, 1000)
drop if n == 1000 & p_min >= 99000
drop if n == 100  & p_min >= 99900
drop if n == 10   & p_min >= 99990
drop p p_max currency widcode n
rename p_min p
rename value a
sort iso year p

save "$work_data/f1_4_temp.dta", replace
 
rsource, terminator(END_OF_R) rpath("$Rpath") roptions(--vanilla)

rm(list = ls())

library(pacman)
p_load(magrittr)
p_load(dplyr)
p_load(readr)
p_load(haven)
p_load(tidyr)
p_load(gpinter)
p_load(purrr)
p_load(stringr)
p_load(glue)
p_load(progress)
options(dplyr.summarise.inform = FALSE)

setwd("change_directory/work-data")
data <- read_dta("change_directory/work-data/f1_4_temp.dta")

gperc <- c(
  0, 50000, 90000, 99000, 99900, 99990, 99999, 99999.9, 99999.99, 99999.999
)

list <- list()
i<- 1
for (iso in c("WO","WO-MER")){
for (y in c(1980,2024)){
  dist <- shares_fit(
    bracketavg = data[data$year==y & data$iso==iso,]$a,
    p = data[data$year==y & data$iso==iso,]$p/1e5,
    fast = TRUE
  )
  tab <- as.data.frame(generate_tabulation(dist, gperc/1e5))
  tab$year <- y
  tab$iso <- iso 
  list[[i]] <- tab
  i <- i+1
}
}
data <- do.call(rbind, list)

write_dta(data, "change_directory/work-data/f1_4_temp2.dta") 

END_OF_R

* get average growth of 10-50% for bottom 50%
use if p>9000 & p<51000 using "$work_data/f1_4_temp.dta", clear
bys iso p (year): gen g_ = (a/a[_n-1])^(1/($year_output-1980))-1
bys iso: egen g = mean(g_)
keep iso g
gen p = 0
duplicates drop iso, force
tempfile bot50g
save `bot50g', replace


clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(npopul) age(992) pop(i) area( WO) year( 1980 $year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"npopul992i")
}
else{
use if widcode == "npopul992i" & (year == $year_output | year == 1980) & (iso == "WO") using $wid2024, clear
}	


keep year value 
ren value npop992i
tempfile adultpop
save `adultpop', replace

use "$work_data/f1_4_temp2.dta", clear

ren fractile p
expand 2 if p == 0, gen(new)
sort iso year p new

gen s = top_share
replace s = bracket_share if (p == 0 & new == 1) | p == .5 

gen a = top_average 
replace a = bracket_average if (p == 0 & new == 1) | p == .5 

gen t = max(threshold,0)
keep iso year p a s t
replace p = -1 if p == 0 & s == 1

bys iso p (year): gen g = (a/a[_n-1])^(1/($year_output-1980))-1

merge m:1 year using `adultpop', nogen

bys iso p (year): gen totg = a*npop*(1-p)-a[_n-1]*npop[_n-1]*(1-p)
replace totg = a*npop-a[_n-1]*npop[_n-1] if p == -1
replace totg = a*npop*0.4-a[_n-1]*npop[_n-1]*0.4 if p == 0.5
replace totg = a*npop*0.5-a[_n-1]*npop[_n-1]*0.5 if p == 0

drop if year == 1980
drop year

replace g = . if p == 0
merge 1:1 iso p using `bot50g', nogen update 

bys iso (p): gen sg = totg/totg[1]


preserve 
keep if iso == "WO"
drop iso totg sg
export excel using "$outputs/WIR_2026_Chapter2.xlsx", sheet("data-F1.4") sheetmod cell(A2) first(var)
putexcel set "$outputs/WIR_2026_Chapter2.xlsx", modify sheet("data-F1.4")
putexcel (B3:B13), nformat(percent)
putexcel (E3:E13), nformat(percent)
restore 




**# Figure 1.5. Income is growing the least for the global middle class 
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
use "$work_data/f1_4_temp2.dta", clear
keep iso year fractile bracket_average
ren (fractile bracket_average) (p aptinc992j)
keep if p>.999999 & iso == "WO"
replace p = p*1e5

tempfile verytop
save `verytop', replace

clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(ahweal aptinc) age(992) pop(j) year(1980 1995 2019 $year_output) area (WO) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"ahweal992j","aptinc992j")
}
else{
use if iso == "WO" & (year == 1980 | year == 1995 | year == 2019 | year == $year_output) & inlist(widcode,"ahweal992j","aptinc992j") using $wid2024, clear
}	

generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
generate long p_max = round(1000*real(regexs(2))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")

generate n = round(p_max - p_min, 1)
keep if inlist(n, 1, 10, 100, 1000)
drop if n == 1000 & p_min >= 99000
drop if n == 100  & p_min >= 99900
drop if n == 10   & p_min >= 99990
drop p p_max currency n
rename p_min p
reshape wide value, i(iso year p) j(widcode) string
renvars value*, predrop(5)

append using `verytop'
sort iso year p

reshape wide ah ap, i(iso p) j(year)

gen elephantw2019 = (ahweal992j2019/ahweal992j1980)^(1/(2019-1980))-1
gen elephantw2024 = (ahweal992j$year_output/ahweal992j1980)^(1/($year_output-1980))-1
gen elephanti2024 = (aptinc992j$year_output/aptinc992j1980)^(1/($year_output-1980))-1
// gen elephanti2024 = (aptinc992j$year_output / aptinc992j1980) - 1

drop a*
gen perc = _n
tsset perc

tssmooth ma elephanti2024s = elephanti2024 if p<99999, w(2 1 2) 
tssmooth ma elephantw2024s = elephantw2024 if p<99999, w(2 1 2) 
tssmooth ma elephantw2019s = elephantw2019 if p<99999, w(2 1 2) 
replace elephanti2024s = elephanti2024[10] if _n<=10
// replace elephanti2024s = elephanti2024 if p>99999


tssmooth ma elephanti2024ss = elephanti2024s, w(4 1 4) 
replace elephanti2024ss = elephanti2024s if p>99999

replace p=p/1000

label var elephanti2024ss "Income growth 1980-2024"
keep p elephanti2024ss
replace elephant = elephant[_N] if _n == _N-2 
drop if p>99.999
replace p = 99.9 if p > 99 & p <= 99.9
replace p = 99.99 if p > 99.9 & p <= 99.99
replace p = 99.999 if p > 99.99 

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.5") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.5")
putexcel (B3:B129), nformat(percent)


* share of growth captured by bottom 50% and top 1%
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(aptinc npopul) age(992)  year(1980 1995 2019 $year_output) p( p0p100 p50p90 p90p100 p0p50 p99p100 ) area (WO) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "aptinc992j", "npopul992i")
}
else{
use if iso == "WO" & (year == 1980 | year == $year_output) & inlist(widcode, "aptinc992j", "npopul992i") & inlist(p, "p0p100", "p50p90", "p90p100", "p0p50", "p99p100") using $wid2024, clear
}	

format value %18.0f

reshape wide value, i(year p) j(widcode) string
renvars value*, predrop(5)
bys year: replace npop = npop[1]
format ap %18.0f
replace ap = ap*npop if p == "p0p100"
replace ap = ap*npop*0.5 if p == "p0p50"
replace ap = ap*npop*0.4 if p == "p50p90"
replace ap = ap*npop*0.1 if p == "p90p100"
replace ap = ap*npop*0.01 if p == "p99p100"

drop npop iso currency
reshape wide aptinc992j, i(p) j(year) 

gen g = aptinc992j2024 - aptinc992j1980
format g %18.0f

gen share = g/g[1]
sum share if p=="p50p90"
loc m40=r(mean)
sum share if p=="p90p100"
loc t10=r(mean)

generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
sort p_min
drop p_min

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.5.sup") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.5.sup")
putexcel (E3:E129), nformat(percent)





**# Figure 1.6.  Wealth is increasing much more at the very top
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(ahweal) age(992) pop(j) area( WO WO-MER ) year( 1995 $year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"ahweal992j")
}
else{
use if widcode == "ahweal992j" & (year == $year_output | year == 1995) & (iso == "WO" | iso == "WO-MER") using $wid2024, clear
}	



generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
generate long p_max = round(1000*real(regexs(2))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")

generate n = round(p_max - p_min, 1)
keep if inlist(n, 1, 10, 100, 1000)
drop if n == 1000 & p_min >= 99000
drop if n == 100  & p_min >= 99900
drop if n == 10   & p_min >= 99990
drop p p_max currency widcode n
rename p_min p
rename value a
sort iso year p

save "$work_data/f1_6_temp.dta", replace
 
rsource, terminator(END_OF_R) rpath("$Rpath") roptions(--vanilla)

rm(list = ls())

library(pacman)
p_load(magrittr)
p_load(dplyr)
p_load(readr)
p_load(haven)
p_load(tidyr)
p_load(gpinter)
p_load(purrr)
p_load(stringr)
p_load(glue)
p_load(progress)
options(dplyr.summarise.inform = FALSE)

setwd("change_directory/work-data")
data <- read_dta("change_directory/work-data/f1_6_temp.dta")

gperc <- c(
  0, 50000, 90000, 99000, 99900, 99990, 99999, 99999.9, 99999.99, 99999.999
)

list <- list()
i<- 1
for (iso in c("WO", "WO-MER")) {
  for (y in c(1995, 2024)) {
    
    # Subset data
    subset <- data[data$year == y & data$iso == iso, ]
    
    # Sort by p
    subset <- subset[order(subset$p), ]
    
    # Clean non-increasing bracket averages
    while (any(diff(subset$a) <= 0)) {
      idx <- which(diff(subset$a) <= 0)[1] + 1
      subset <- subset[-idx, ]
    }
    
    # Fit distribution
    dist <- shares_fit(
      bracketavg = subset$a,
      p = subset$p / 1e5,
      fast = TRUE
    )
    
    tab <- as.data.frame(generate_tabulation(dist, gperc / 1e5))
    tab$year <- y
    tab$iso <- iso
    
    list[[i]] <- tab
    i <- i + 1
  }
}
data <- do.call(rbind, list)

write_dta(data, "change_directory/work-data/f1_6_temp2.dta")

END_OF_R

* get average growth of 10-50% for bottom 50%
use if p>9000 & p<51000 using "$work_data/f1_6_temp.dta", clear
bys iso p (year): gen g_ = (a/a[_n-1])^(1/($year_output-1995))-1
bys iso: egen g = mean(g_)
keep iso g
gen p = 0
duplicates drop iso, force
tempfile bot50g
save `bot50g', replace


clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(npopul) age(992) pop(i) area( WO) year( 1995 $year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"npopul992i")
}
else{
use if widcode == "npopul992i" & (year == $year_output | year == 1995) & (iso == "WO") using $wid2024, clear
}	


keep year value 
ren value npop992i
tempfile adultpop
save `adultpop', replace

use "$work_data/f1_6_temp2.dta", clear

ren fractile p
expand 2 if p == 0, gen(new)
sort iso year p new

gen s = top_share
replace s = bracket_share if (p == 0 & new == 1) | p == .5 

gen a = top_average 
replace a = bracket_average if (p == 0 & new == 1) | p == .5 

gen t = max(threshold,0)
keep iso year p a s t
replace p = -1 if p == 0 & s == 1

bys iso p (year): gen g = (a/a[_n-1])^(1/($year_output-1995))-1

merge m:1 year using `adultpop', nogen

bys iso p (year): gen totg = a*npop*(1-p)-a[_n-1]*npop[_n-1]*(1-p)
replace totg = a*npop-a[_n-1]*npop[_n-1] if p == -1
replace totg = a*npop*0.4-a[_n-1]*npop[_n-1]*0.4 if p == 0.5
replace totg = a*npop*0.5-a[_n-1]*npop[_n-1]*0.5 if p == 0

drop if year == 1995
drop year

replace g = . if p == 0
merge 1:1 iso p using `bot50g', nogen update 

bys iso (p): gen sg = totg/totg[1]

*Correct the very very top using forbes Raw data
append using $forbes
replace p=.9999999 if perc=="top0.00001"
replace p=.99999999 if perc=="top0.000001"
replace npop992i=npopul992i if npop992i==.
drop if inlist(p,.9999999,.99999999) & perc==""
sort iso p
drop npopul992i perc source

preserve 
keep if iso == "WO"
drop iso totg sg
export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.6") sheetmod cell(A2) first(var)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.6")
putexcel (B3:B13), nformat(percent)
putexcel (E3:E13), nformat(percent)
restore 

keep p iso sg
replace iso = substr(iso,4,3)
replace iso = "PPP" if iso == ""
reshape wide sg, i(p) j(iso "PPP" "MER") string
g year=$year_output

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.6") sheetmod cell(A2) first(var)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.6")
putexcel (B3:C13), nformat(percent)





**# Figure 1.7. Wealth grows faster among the very wealthy 
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
use $forbes, clear
sum g if perc=="top0.00001" & iso=="WO"
loc Top1_10million=r(mean)
sum g if perc=="top0.000001" & iso=="WO"
loc Top1_100million=r(mean)

use "$work_data/f1_6_temp2.dta", clear
keep iso year fractile bracket_average
ren (fractile bracket_average) (p ahweal992j)
keep if p>.999999 & iso == "WO"
replace p = p*1e5

tempfile verytop
save `verytop', replace

clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(ahweal aptinc) age(992) pop(j) year(1980 1995 2019 $year_output) area (WO) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"ahweal992j","aptinc992j")
}
else{
use if iso == "WO" & (year == 1980 | year == 1995 | year == 2019 | year == $year_output) & inlist(widcode,"ahweal992j","aptinc992j") using $wid2024, clear
}	


generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
generate long p_max = round(1000*real(regexs(2))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")

generate n = round(p_max - p_min, 1)
keep if inlist(n, 1, 10, 100, 1000)
drop if n == 1000 & p_min >= 99000
drop if n == 100  & p_min >= 99900
drop if n == 10   & p_min >= 99990
drop p p_max currency n
rename p_min p
reshape wide value, i(iso year p) j(widcode) string
renvars value*, predrop(5)

append using `verytop'
sort iso year p

reshape wide ah ap, i(iso p) j(year)

gen elephantw2019 = (ahweal992j2019/ahweal992j1995)^(1/(2019-1995))-1
gen elephantw2024 = (ahweal992j2024/ahweal992j1995)^(1/(2024-1995))-1
gen elephanti2024 = (aptinc992j2024/aptinc992j1980)^(1/(2024-1980))-1

drop a*
gen perc = _n
tsset perc
tssmooth ma elephanti2024s = elephanti2024 if p<99999, w(2 1 2) 
tssmooth ma elephantw2024s = elephantw2024 if p<99999, w(2 1 2) 
tssmooth ma elephantw2019s = elephantw2019 if p<99999, w(2 1 2) 
replace elephantw2024s = elephantw2024[10] if _n<=10
replace elephantw2024s = elephantw2024 if p>99999

tssmooth ma elephantw2024ss = elephantw2024s, w(4 1 4) 
replace elephantw2024ss = elephantw2024 if p>99998

replace p=99999.9 if p==99999.99
insobs 1
replace p =99999.99 if iso==""
sort p
replace iso="WO" if p==99999.99 

*replace the very very top using forbes raw data
replace elephantw2024ss=`Top1_10million' if p==99999.99 
replace elephantw2024ss=`Top1_100million' if p==99999.999 


replace p=p/1000


label var elephantw2024ss "Wealth growth 1995-2024"
keep p elephantw2024ss
// replace elephant = elephant[_N] if _n == _N-2 
drop if p>99.999999
replace p = 99.9 if p > 99 & p <= 99.9
replace p = 99.99 if p > 99.9 & p <= 99.99
replace p = 99.999 if p > 99.99 & p <= 99.999



export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.7") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.7")
putexcel (B3:B132), nformat(percent)


* share of growth captured by bottom 50% and top 1%
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(ahweal npopul) age(992)  year(1980 1995 2019 $year_output) p( p0p100 p50p90 p90p100 p0p50 p99p100 ) area (WO) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "ahweal992j", "npopul992i")
}
else{
use if iso == "WO" & (year == 1995 | year == $year_output) & inlist(widcode, "ahweal992j", "npopul992i") & inlist(p, "p0p100", "p50p90", "p90p100", "p0p50", "p99p100") using $wid2024, clear
}	


reshape wide value, i(year p) j(widcode) string
renvars value*, predrop(5)
bys year: replace npop = npop[1]
replace ah = ah*npop if p == "p0p100"
replace ah = ah*npop*0.5 if p == "p0p50"
replace ah = ah*npop*0.4 if p == "p50p90"
replace ah = ah*npop*0.1 if p == "p90p100"
replace ah = ah*npop*0.01 if p == "p99p100"

drop npop iso currency
reshape wide ahweal992j, i(p) j(year) 

gen g = ahweal992j2024 - ahweal992j1995
gen share = g/g[1]

generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
sort p_min
drop p_min

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.7.sup") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.7.sup")
putexcel (E3:E129), nformat(percent)



**# Figure 1.8. Extreme wealth inequality is persistent and increasing
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(shweal) age(992)  year(1995 $year_output) p( p0p50 p99.999p100 ) area (WO) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "shweal992j")
}
else{
use if iso == "WO" & inrange(year,1995,$year_output) & widcode == "shweal992j" & inlist(p, "p0p50", "p99.999p100") using $wid2024, clear
}	

replace p = "bot50"  if p == "p0p50"
replace p = "top001" if p == "p99.999p100"

reshape wide value, i(year) j(p) string
renvars value*, predrop(5)
keep year bot50 top001

label var bot50  "Bottom 50%"
label var top001 "Top 0.001%"

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.8") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.8")
putexcel (B3:C33), nformat(percent)



**# Figure 1.9. and Figure 1.11
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(sptinc) age(992) pop(j) area( WO ) p(p0p50 p50p90 p90p100 p99p100 p99.9p100 p99.99p100 p99.999p100) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"sptinc992j")
}
else{
use if inlist(iso, "WO") &  ///
inlist(widcode, "sptinc992j") ///
	  & inlist(p, "p0p50", "p50p90", "p90p100", "p99p100", "p99.9p100", "p99.99p100", "p99.999p100")	using $wid2024, clear
}	

drop currency iso widcode

greshape wide value, i(year) j(p) string

keep if inlist(year,1820,1850,1880,1900,1910,1920,1930,1940,1950,1960,1970,1980,1985,1990,1995,1997,2000,2002,2005,2007,2010,2015,2017,2020) | inrange(year,2022, $year_output)


renvars value*, pred(5)
label var p0p50   "Bottom 50%"
label var p50p90  "Middle 40%"
label var p90p100 "Top 10%"
label var p99p100 "Top 1%"
label var p99_9p100 "Top 0.1%"
label var p99_99p100 "Top 0.01%"
label var p99_999p100 "Top 0.001%"

tsset year
tempfile f2_1
save `f2_1',replace

keep year p0p50 p50p90 p90p100
export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.9") sheetmod cell(A1) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.9")
putexcel (B2:D29), nformat(percent)


**# Figure 1.10. and Figure 1.11
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
use `f2_1',clear
keep year p0p50 p99p100  p99_9p100
export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.10") sheetmod cell(A1) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.10")
putexcel (B2:D29), nformat(percent)


**# Figure 1.12. There is very large inequality across regions 
* ---------------------------------------------------------------------------- 
* ---------------------------------------------------------------------------- *
 
 clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(ahweal anninc) age(992) area( WO XF QE XB XL XN XR XS QL) p(p0p100) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "ahweal992j", "anninc992i") 
}
else{
use if (inlist(iso, "WO", "XF", "QE", "XB", "XL") | inlist(iso, "XN", "XR", "XS", "QL")) & ///
	inlist(widcode, "ahweal992j", "anninc992i") & year == $year_output & p == "p0p100" ///
	using $wid2024, clear
}	
 


replace iso = "Sub-Saharan Africa"      if iso == "XF"
replace iso = "Europe"                  if iso == "QE"
replace iso = "North America & Oceania"           if iso == "XB"
replace iso = "Latin America"           if iso == "XL"
replace iso = "MENA"                    if iso == "XN"
replace iso = "Russia & Central Asia"   if iso == "XR"
replace iso = "South & Southeast Asia" if iso == "XS"
replace iso = "East Asia"               if iso == "QL"

rename iso RegionWID
drop year p currency
reshape wide value, i(RegionWID) j(widcode) string
renvars value*, pred(5)

generate aa = anninc992i if RegionWID == "WO"
egen ainc_wo = mode(aa)
drop aa

generate aa = ahweal992j if RegionWID == "WO"
egen aweal_wo = mode(aa)
drop aa

drop if RegionWID == "WO"

generate average_inc  = anninc992i/ainc_wo
generate average_weal = ahweal992j/aweal_wo

gsort average_weal
keep RegionWID average_inc average_weal
export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.12") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter1.xlsx", modify sheet("data-F1.12")
putexcel (B3:C10), nformat(percent)	







**# Figure 1.13. There is also very large inequality within regions
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
 clear 
insobs 1 
g use_widcode="$use_widcode"
	

if use_widcode=="YES"{
	wid, ind(ahweal) age(992) pop(j) area(XN-PPP XF-PPP QE-PPP XB-PPP XL-PPP XR-PPP XS-PPP QL-PPP) p(p0p50 p50p90 p90p100 ) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "ahweal992j") 
}
 else{
use if widcode == "ahweal992j" & inlist(iso, "XN-PPP", "XF-PPP", "QE-PPP", "XB-PPP", "XL-PPP", "XR-PPP", "XS-PPP", "QL-PPP") & ///
       year == $year_output & inlist(p, "p0p50", "p50p90", "p90p100") using $wid2024, clear
} 

replace iso = "Sub-Saharan Africa"      if iso == "XF-PPP"
replace iso = "Europe"                  if iso == "QE-PPP"
replace iso = "North America & Oceania"        if iso == "XB-PPP"
replace iso = "Latin America"           if iso == "XL-PPP"
replace iso = "MENA"                    if iso == "XN-PPP"
replace iso = "Russia & Central Asia"   if iso == "XR-PPP"
replace iso = "South & Southeast Asia" if iso == "XS-PPP"
replace iso = "East Asia"               if iso == "QL-PPP"


drop widcode currency 
replace p = "bot50" if p == "p0p50"
replace p = "top10" if p == "p90p100"
replace p = "mid40" if p == "p50p90"

replace value = round(value, 0.0001) 

reshape wide value, i(iso) j(p) string
renvars value*, predrop(5)
label var bot50 "Bottom 50%"
label var mid40 "Middle 40%"
label var top10 "Top 10%"

order year iso bot50 mid40 top10 
keep  year iso bot50 mid40 top10 
 
gen T10B50 = top10/(bot50) 

keep year iso T10B50 
sort T10B50

export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.13") sheetmod cell(A1) first(var)

tempfile original
save `original',replace
use if inlist(widcode,"rhweal992j","rptinc992j") & inlist(iso, "XN-PPP", "XF-PPP", "QE-PPP", "XB-PPP", "XL-PPP", "XR-PPP", "XS-PPP", "QL-PPP") & ///
       year == $year_output  using $wid2024, clear
drop p year currency
	   greshape wide value, i(iso) j(widcode) string
renvars value*, predrop(5)	   
replace iso = "Sub-Saharan Africa"      if iso == "XF-PPP"
replace iso = "Europe"                  if iso == "QE-PPP"
replace iso = "North America & Oceania"        if iso == "XB-PPP"
replace iso = "Latin America"           if iso == "XL-PPP"
replace iso = "MENA"                    if iso == "XN-PPP"
replace iso = "Russia & Central Asia"   if iso == "XR-PPP"
replace iso = "South & Southeast Asia" if iso == "XS-PPP"
replace iso = "East Asia"               if iso == "QL-PPP"
label var rptinc992j "T10B50 income"
label var rhweal992j "T10B50 wealth"
sort rhweal992j
merge 1:1 iso using `original'
drop _merge T10B50
sort rhweal992j
order year iso rhweal992j  rptinc992j
export excel using "$outputs/WIR_2026_Chapter1.xlsx", sheet("data-F1.13") sheetmod cell(A1) first(var)




**# Box 1.2: Inequality Transparency Index
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *

use if 	 inrange(year,2022,2022) & p=="p0p100" & ///
		inlist(widcode, "iquali999i") ///
		using $wid2024, clear	
keep  iso value	

preserve
import excel using "$raw/country-codes-new.xlsx", firstrow clear
renvars code shortname / iso isoname
keep iso isoname 
replace isoname="Turkiye" if iso=="TR"
replace isoname="Czechia" if iso=="CZ"
replace isoname="Russia" if iso=="RU"
replace isoname="Eswatini" if iso=="SZ"
replace isoname="Syria" if iso=="SY"
replace isoname="South Korea" if iso=="KR"
tempfile isoname 
save `isoname', replace
restore

merge m:1 iso using `isoname', nogen keep(master matched) keepusing(isoname)

save "$work_data/transparency_map.dta", replace
