********************************************************************************
********************************************************************************
* Date: November 2025
** Do: Code to reproduce Chapter 3 of World Inequality Report 2026
*** Author: Ricardo Gómez-Carrera
	* 1. Figure 3.1. Global wealth has expanded dramatically over the past three decades 
	* 2. Figure 3.2. From the post-war decades onward, most countries experienced renewed rises in wealth ratios
	* 3. Figure 3.3. Domestic capital remains the foundation of national wealth everywhere
	* 4. Figure 3.4. Since the 1970s, North America & Oceania has shifted into the largest net debtor
	* 5. Figure 3.5. The rise of private wealth and the decline of public wealth in every region 
	* 6. Figure 3.6. The rising capital share in global income
	* 7. Figure 3.7. Most of the global population is clustered at low levels of wealth
	* 8. Figure 3.8. Extreme wealth inequality is high in all regions
	* 9. Figure 3.9.a. The geography of the wealthiest has diversified, especially toward East Asia
	*10. Figure 3.10. Bottom 50% shares are small everywhere
	*11. Figure 3.11. Middle 40% shares are small almost everywhere
	*12. Figure 3.12. Top 10% wealth shares are large everywhere
	*13. Figure 3.13. Top 1% wealth shares are very large
********************************************************************************
********************************************************************************

/*
For information on how to use the widcode and the World Inequality Database, visit:
	*1. https://wid.world/codes-dictionary/
	*2. https://github.com/world-inequality-database/wid-stata-tool
	*3. https://github.com/world-inequality-database/wid-r-tool
	*4. https://wid.world/document/distributional-national-accounts-dina-guidelines-2025-methods-and-concepts-used-in-the-world-inequality-database/
	
	For other data inputs visit:
	*1. https://github.com/world-inequality-database/wid-world/tree/master/data-input
	
*/



**# Figure 3.1. Global wealth has expanded dramatically over the past three decades
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
use if 	 inrange(year,2024,2024) & p=="p0p100" & ///
		inlist(widcode, "xlceup999i") ///
		using $wid2024, clear
gen maincountry=.
foreach c of global maincountries {
	replace maincountry=1 if iso=="`c'"
	}
keep if (maincountry==1 )		
keep year iso value	widcode	
greshape wide value, i(year iso) j(widcode) string
renvars value*, pred(5)
drop year
tempfile ppp
save `ppp',replace
		

use if 	inlist(widcode, "mnweal999i", "npopul999i") ///
		using $wid2024, clear
gen maincountry=.
foreach c of global maincountries {
	replace maincountry=1 if iso=="`c'"
	}
keep if (maincountry==1 )
keep if year>=1995		
keep year iso value	widcode	
greshape wide value, i(year iso) j(widcode) string
renvars value*, pred(5)
merge m:1 iso using `ppp'
drop _merge
replace mnweal999i=(mnweal999i/xlceup999i) 
drop xlceup999i

gen region2=""
	foreach c of global EURO {
		replace region2="QE" if iso=="`c'"
		}
	foreach c of global NAOC {
		replace region2="XB" if iso=="`c'"
		}
	foreach c of global LATA {
		replace region2="XL" if iso=="`c'"
		}
	foreach c of global MENA {
		replace region2="XN" if iso=="`c'"
		}
	foreach c of global SSAF {
		replace region2="XF" if iso=="`c'"
		}
	foreach c of global RUCA {
		replace region2="XR" if iso=="`c'"
		}
	foreach c of global EASA {
		replace region2="QL" if iso=="`c'"
		}	
	foreach c of global SSEA {
		replace region2="XS" if iso=="`c'"
		}		
replace region2="QE" if inlist(iso, "QM","OC")
replace region2="XB" if inlist(iso, "OH") 		
replace region2="XL" if inlist(iso, "OD")
replace region2="XN" if inlist(iso, "OE")
replace region2="XF" if inlist(iso, "OJ")
replace region2="XR" if inlist(iso, "OA")
replace region2="QL" if inlist(iso, "OB")
replace region2="XS" if inlist(iso, "OI") 		
	ren region2 region
misstable summarize	
replace mnweal999i =mnweal999i /1000000000
collapse (sum)	mnweal999i ,by(year region) 
*World
preserve
collapse (sum)	mnweal999i ,by(year ) 
g region ="WO"

tempfile world_nnw
save `world_nnw',replace
restore

append using 	`world_nnw'

greshape wide mnweal999i, i(year) j(region) string
renvars mnweal999i*, pred(10)

label var QE "Europe"
label var QL "East Asia"
label var XB "North America & Oceania"
label var XF "Sub-Saharan Africa"
label var XL "Latin America"
label var XN "MENA"
label var XR "Russia & Central Asia"
label var XS "South & Southeast Asia"
label var WO "World"
tsset year
tsline QE QL XB XF XL XN XR XS WO

 order  year QE QL XB XF XL XN XR XS WO		

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.1") sheetmod cell(A2) first(varl)



**# Figure 3.2. From the post-war decades onward, most countries experienced renewed rises in wealth ratios
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
use if widcode == "wwealn999i" ///
     using $wid2024, clear
	    drop widcode currency p
tab iso if year<1980		
	
	gl hist "AU CA CN DE ES FR GB IN IT JP SE US ES CH NL WO"
	g keep=""
	foreach c of global hist {
		replace keep="yes" if iso=="`c'"
		}
	keep if keep =="yes"
	drop keep
	
greshape wide  value, i(year) j(iso) string	
renvars value*, predrop(5)
order year 
label var AU "Australia" 
label var CA "Canada" 
label var CN "China" 
label var DE "Germany" 
label var FR "France"
label var GB "United Kingdom"
label var IN "India"
label var IT "Italy"
label var JP "Japan"
label var SE "Sweden"
label var US "United States"
label var ES "Spain" 
label var CH "Switzerland" 
label var NL "Netherlands" 
label var WO "World" 

tsset year
*Interpolate missing values in between for graphical purposes
foreach c of global hist {
	sum year if `c'!=.
	loc min=r(min)
	dis `min'
	loc min=r(min)
	ipolate `c' year if inrange(year,r(min),r(max)), gen(interpolation) epolate
	replace `c'=interpolation if inrange(year,r(min),r(max)) & `c'==.
	drop interpolation
}	

	keep year CN DE FR GB IN  JP SE US  ES WO 
	order year CN DE FR GB IN  JP SE US  ES WO 

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.2") sheetmod cell(A2) first(varl)	



**# Figure 3.3. Domestic capital remains the foundation of national wealth everywhere
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 

use if 	inlist(widcode, "mnweal999i","mnninc999i", "mnwdka999i","mnwnxa999i") ///
		using $wid2024, clear
gen maincountry=.
foreach c of global maincountries {
	replace maincountry=1 if iso=="`c'"
	}
keep if (maincountry==1 )
keep if year>=1995		
keep year iso value	widcode	
greshape wide value, i(year iso) j(widcode) string
renvars value*, pred(5)
replace mnwnxa999i=mnweal999i-mnwdka999i
merge m:1 iso using `ppp'
drop _merge
foreach var in mnweal999i mnninc999i mnwdka999i mnwnxa999i{
	replace `var'=(`var'/xlceup999i) /1000000000
}

drop xlceup999i

gen region2=""
	foreach c of global EURO {
		replace region2="QE" if iso=="`c'"
		}
	foreach c of global NAOC {
		replace region2="XB" if iso=="`c'"
		}
	foreach c of global LATA {
		replace region2="XL" if iso=="`c'"
		}
	foreach c of global MENA {
		replace region2="XN" if iso=="`c'"
		}
	foreach c of global SSAF {
		replace region2="XF" if iso=="`c'"
		}
	foreach c of global RUCA {
		replace region2="XR" if iso=="`c'"
		}
	foreach c of global EASA {
		replace region2="QL" if iso=="`c'"
		}	
	foreach c of global SSEA {
		replace region2="XS" if iso=="`c'"
		}		
replace region2="QE" if inlist(iso, "QM","OC")
replace region2="XB" if inlist(iso, "OH") 		
replace region2="XL" if inlist(iso, "OD")
replace region2="XN" if inlist(iso, "OE")
replace region2="XF" if inlist(iso, "OJ")
replace region2="XR" if inlist(iso, "OA")
replace region2="QL" if inlist(iso, "OB")
replace region2="XS" if inlist(iso, "OI") 		
	ren region2 region
misstable summarize	

collapse (sum)	mnninc999i mnwdka999i mnweal999i mnwnxa999i ,by(year region) 
*World
preserve
collapse (sum)	mnninc999i mnwdka999i mnweal999i mnwnxa999i,by(year ) 
g region ="WO"

tempfile world_nnw
save `world_nnw',replace
restore

append using 	`world_nnw'

foreach var in mnweal999i  mnwdka999i mnwnxa999i{
	replace `var'=(`var'/mnninc999i) 
}
drop mnninc999i
drop if region=="WO"

label var mnwdka999i "Domestic Capital"
label var mnweal999i "Net National Wealth"
label var mnwnxa999i "Net Foreign Assets"

 sort region year 
  
replace region = "Europe" if region == "QE"
replace region = "East Asia" if region == "QL"
replace region = "North America & Oceania" if region == "XB"
replace region = "Sub-Saharan Africa" if region == "XF"
replace region = "Latin America" if region == "XL"
replace region = "MENA" if region == "XN"
replace region = "Russia & Central Asia" if region == "XR"
replace region = "South & Southeast Asia" if region == "XS"
replace region = "World" if region == "WO"

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.3") sheetmod cell(A2) first(varl)


**# Figure 3.4. Since the 1970s, North America & Oceania has shifted into the largest net debtor
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Nievas and Piketty (2025)
Figures 1 and 15 in Nievas and Piketty (2025)

Paper
https://wid.world/document/unequal-exchange-and-north-south-relations-evidence-from-global-trade-flows-and-the-world-balance-of-payments-1800-2025-world-inequality-lab-working-paper-2025-11/

Replication package
https://wid.world/document/data-file-to-unequal-exchange-and-north-south-relations-evidence-from-global-trade-flows-and-the-world-balance-of-payments-1800-2025-world-inequality-lab-working-paper-2025-11/
*/


**# Figure 3.5.a. The rise of private wealth and the decline of public wealth in every region 
* ---------------------------------------------------------------------------- *
* Private
* ---------------------------------------------------------------------------- *
use if widcode == "wwealp999i" & inlist(iso, "WO", "XN", "XF", "QE", "XB", "XL", "XR", "XS", "QL") ///
 using $wid2024, clear
	    drop widcode currency p
		
greshape wide  value, i(year) j(iso) string	
renvars value*, predrop(5)
order year WO
label var XF "Sub-Saharan Africa" 
label var XL "Latin America" 
label var XN "MENA" 
label var XR "Russia & Central Asia" 
label var QE "Europe" 
label var XB "North America & Oceania"
label var QL "East Asia"
label var XS "South & Southeast Asia"
label var WO "World"

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.5.a") sheetmod cell(A2) first(varl)


**# Figure 3.5.b. The rise of private wealth and the decline of public wealth in every region
* ---------------------------------------------------------------------------- *
*Public
* ---------------------------------------------------------------------------- *
use if widcode == "wwealg999i" & inlist(iso, "WO", "XN", "XF", "QE", "XB", "XL", "XR", "XS", "QL") ///
 using $wid2024, clear
	    drop widcode currency p
		

greshape wide  value, i(year) j(iso) string	
renvars value*, predrop(5)
order year WO
label var XF "Sub-Saharan Africa" 
label var XL "Latin America" 
label var XN "MENA" 
label var XR "Russia & Central Asia" 
label var QE "Europe" 
label var XB "North America & Oceania"
label var QL "East Asia"
label var XS "South & Southeast Asia"
label var WO "World"
tsset year

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.5.b.") sheetmod cell(A2) first(varl)


**# Figure 3.6. The rising capital share in global income
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Dietrich et al. (2025)
Figures in Dietrich et al. (2025):
- (page 59/66) Labor Share in the total Economy (% of factor-price GDP) by World Region
- (page 61/66) Gross Capital Share in the total Economy (% of factor-price GDP) by World Region

Document
https://wid.world/document/extending-wid-national-accounts-series-institutional-sectors-and-factor-shares-world-inequality-lab-technical-note-2025-03/

Replication package
https://wid.world/document/data-file-to-extending-wid-national-accounts-series-institutional-sectors-and-factor-shares-appendix-to-wil-technical-note-2025-03/
*/




**# Figure 3.7. Most of the global population is clustered at low levels of wealth
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *

use "$work_data/temp_density.dta", clear //from Figure 2.5
keep dweal expdweal weal* 
renvars weal*, predrop(4)

replace expdweal = round(expdweal,10) 
replace expdweal = round(expdweal,50) if expdweal>=200
replace expdweal = round(expdweal,100) if expdweal>=1000
replace expdweal = round(expdweal,500) if expdweal>=2000
replace expdweal = round(expdweal,5000) if expdweal>=10000
replace expdweal = round(expdweal,50000) if expdweal>=100000
replace expdweal = round(expdweal,500000) if expdweal>=1000000

label var XF "Sub-Saharan Africa" 
label var XL "Latin America" 
label var XN "MENA" 
label var XR "Russia & Central Asia" 
label var QE "Europe" 
label var XB "North America & Oceania"
label var QL "East Asia"
label var XS "South & Southeast Asia"

order d expd XB QE XR XN XL XF QL XS

line XB QE XR XN XL XF QL XS d

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.7") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter3.xlsx", modify sheet("data-F3.7")
putexcel (C3:J202), nformat(percent)


**# Figure 3.8. Extreme wealth inequality is high in all regions
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
 clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, ind(shweal) age(992) pop(j) area(XN XF QE XB XL XR XS QL) p(p0p50 p50p90 p90p100 p99p100 ) year($year) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "shweal992j") 
}
else{
use if widcode == "shweal992j" & inlist(iso, "XN", "XF", "QE", "XB", "XL", "XR", "XS", "QL") ///
		& year == $year_output & inlist(p, "p0p50", "p50p90", "p90p100", "p99p100" ) using $wid2024, clear
}	

 
replace iso = "Sub-Saharan Africa"      if iso == "XF"
replace iso = "Europe"                  if iso == "QE"
replace iso = "North America & Oceania" if iso == "XB"
replace iso = "Latin America"           if iso == "XL"
replace iso = "MENA"                    if iso == "XN"
replace iso = "Russia & Central Asia"   if iso == "XR"
replace iso = "South & Southeast Asia" if iso == "XS"
replace iso = "East Asia"               if iso == "QL"

drop widcode currency 
replace p = subinstr(p, ".", "_", .)
reshape wide value, i(iso) j(p) string
renvars value*, predrop(5)
label var p0p50   "Bottom 50%"
label var p50p90  "Middle 40%"
label var p90p100 "Top 10%"
label var p99p100 "Top 1%"


order year iso p0p50 p50p90 p90p100 p99p100 
gsort p90p100
export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.8") sheetmod cell(A1) first(varl)
putexcel set "$outputs/WIR_2026_Chapter3.xlsx", modify sheet("data-F3.8")
putexcel (C2:J9), nformat(percent)






**# Figure 3.9 The geography of the wealthiest has diversified, especially toward East Asia
* ---------------------------------------------------------------------------- *
*Change year if necessary
* ---------------------------------------------------------------------------- *

 clear 
insobs 1 
g use_widcode="$use_widcode"
	

if use_widcode=="YES"{
	wid, ind(ahweal) age(992) pop(j) area(QE XB XR XL XF XN CN IN) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "ahweal992j")
}
else{
use if inlist(widcode, "ahweal992j") & year==$year_output using $wid2024, clear
drop if missing(value)
keep if inlist(iso, "QE", "XB", "XR", "XL") ///
	  | inlist(iso, "XF", "XN", "CN", "IN")
}

// Parse percentiles
generate long p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")
generate long p_max = round(1000*real(regexs(2))) if regexm(p, "^p([0-9\.]+)p([0-9\.]+)$")

replace p_min = round(1000*real(regexs(1))) if regexm(p, "^p([0-9\.]+)$")

replace p_max = 1000*100 if missing(p_max)

replace p_max = p_min + 1000 if missing(p_max) & inrange(p_min, 0, 98000)
replace p_max = p_min + 100  if missing(p_max) & inrange(p_min, 99000, 99800)
replace p_max = p_min + 10   if missing(p_max) & inrange(p_min, 99900, 99980)
replace p_max = p_min + 1    if missing(p_max) & inrange(p_min, 99990, 99999)

replace p = "p" + string(round(p_min/1e3, 0.001)) + "p" + string(round(p_max/1e3, 0.001)) if !missing(p_max)

// Keep only g-percentiles
generate n = round(p_max - p_min, 1)
keep if inlist(n, 1, 10, 100, 1000)
drop if n == 1000 & p_min >= 99000
drop if n == 100  & p_min >= 99900
drop if n == 10   & p_min >= 99990
drop p p_max currency
rename p_min p
duplicates drop iso year p widcode, force
sort iso year widcode p

reshape wide value, i(iso year p) j(widcode) string

rename valueahweal992j a

preserve 
 clear 
insobs 1 
g use_widcode="$use_widcode"


if use_widcode=="YES"{
	wid, ind(xlceup) age(999) pop(i) area(CN IN) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "xlceup999i")
}
else{
	use if widcode == "xlceup999i" & inlist(iso, "CN", "IN") & year == $year_output using $wid2024, clear
}	

	drop p widcode currency year
	rename value ppp
	
	tempfile ppp
	save `ppp'
restore 
merge m:1 iso using `ppp', nogen

replace a = a/ppp if !missing(ppp)
drop ppp

keep if year == $year_output


// rank and compute new bracket for each region
preserve 
 clear 
insobs 1 
g use_widcode="$use_widcode"
	

if use_widcode=="YES"{
	wid, ind(npopul) age(992) pop(i) area(QE XB XR XL XF XN CN IN) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "npopul992i")
}
else{
		use if widcode == "npopul992i" & year == $year_output using $wid2024, clear
	keep if inlist(iso, "QE", "XB", "XR", "XL") ///
		  | inlist(iso, "XF", "XN", "CN", "IN")
}

	drop p widcode currency
	rename value npopul992i
	
	tempfile npopul992i
	save `npopul992i'
restore 
merge m:1 iso year using `npopul992i', nogen
// aqui me quede
preserve 
	use "$work_data/OtherAsia-ptinc-hweal", clear //from Figure 2.7
	keep year iso p ai npopul992i
	rename ai a
	keep if year == $year_output
	
	tempfile OAsia
	save `OAsia'
restore
append using `OAsia'
drop n

// recompute size of the bracket
gen n=0.01
replace n=0.001 if p>=99000 & p<99900
replace n=0.0001 if p>=99900 & p<99990
replace n=0.00001 if p>=99990

generate pop = n*npopul992i

gsort year -a
bys year : generate rank = sum(pop)
bys year : replace rank = 1e5*(1 - rank/rank[_N])

egen bracket = cut(rank), at(0(1000)99000 99100(100)99900 99910(10)99990 99991(1)99999 200000)

collapse (sum) pop, by(year bracket iso)
bys year bracket : egen tot = sum(pop)
gen share = (pop/tot)*100
// gen share = (pop/tot)
keep iso year bracket share 
reshape wide share, i( year bracket) j(iso) string
ds year bracket, not
foreach l in `r(varlist)' {
replace `l' = 0 if missing(`l')
}

bys year : gen perc = _n
tsset year perc
ds year bracket, not

foreach x in `r(varlist)' {
	lowess `x' perc, bwidth(.125) gen(`x'_sm)
}


ren bracket p 


keep p share*sm
foreach i in XF CN IN XL XN XR QE XB OA{
	replace share`i'_sm = share`i'_sm/100
}
label var shareIN_sm "India" 
label var shareOA_sm "Other Asia" 
label var shareCN_sm "China" 
label var shareXF_sm "SSA" 
label var shareXL_sm "Latin America" 
label var shareXN_sm "MENA" 
label var shareXR_sm "Russia & Central Asia" 
label var shareQE_sm "Europe" 
label var shareXB_sm "North America & Oceania"

* just for correct graduations on graph
replace p = p/1000
replace p = . if mod(_n-1,10)!=0 

replace p = round(floor(p),.1) if p>99 & p<99.91 
replace p = round(floor(p*10)/10,.01) if p>=99.91 & p<99.991 
replace p = round(floor(p*100)/100,.001) if p>=99.991 

order p shareIN_sm shareOA_sm shareCN_sm shareXF_sm shareXL_sm shareXN_sm shareXR_sm shareQE_sm shareXB_sm    

export excel using "$outputs/WIR_2026_Chapter3.xlsx", sheet("data-F3.9") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter3.xlsx", modify sheet("data-F3.9")
putexcel (B3:J129), nformat(percent)



**# F3.10-3.14
* ---------------------------------------------------------------------------- 
* ---------------------------------------------------------------------------- *
import excel using "$raw/country-codes-new.xlsx", firstrow clear
renvars code shortname / iso isoname
keep iso isoname 
replace isoname="Turkiye" if iso=="TR"
replace isoname="Czechia" if iso=="CZ"
replace isoname="Russia" if iso=="RU"
replace isoname="Eswatini" if iso=="SZ"
replace isoname="Syria" if iso=="SY"
replace isoname="South Korea" if iso=="KR"
tempfile isoname 
save `isoname', replace

 clear 
insobs 1 
g use_widcode="$use_widcode"


if use_widcode=="YES"{
	wid, ind(shweal) age(992) pop(j) p( p0p50 p90p100 p99p100 ) year($year_output) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"shweal992j") 
}
else{
use if widcode == "shweal992j" & ///
       year == $year_output  & inlist(p, "p0p50", "p90p100", "p99p100") using $wid2024, clear
}	


drop widcode currency 
replace p = "bot50" if p == "p0p50"
replace p = "top10" if p == "p90p100"
replace p = "top1"  if p == "p99p100"

reshape wide value, i(iso) j(p) string
renvars value*, predrop(5)


merge m:1 iso using `isoname', nogen keep(master matched) keepusing(isoname)

preserve
use if inlist(widcode,"rhweal992j") & year == $year_output using $wid2024, clear
keep iso value
ren value T10B50
tempfile bd2
save `bd2', replace
restore 


merge 1:1 iso using `bd2' , nogen keep(master matched) 

g corecountry=.
foreach country in $corecountries{
	replace corecountry=1 if iso=="`country'"
}
keep if corecountry==1
drop corecountry

keep iso bot50  top1 top10 year T10B50  isoname
order iso bot50  top1 top10 year T10B50  isoname

save "$work_data/T10B50_wealth_map.dta", replace